<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 */
class AR_Display_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * API Key instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      AR_Display_Api_Key    $api_key    API Key instance.
     */
    private $api_key;

    /**
     * Settings instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      AR_Display_Settings    $settings    Settings instance.
     */
    private $settings;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Load the product list table class
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-product-list-table.php';
        
        // Load and initialize product metabox
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-product-metabox.php';
        new AR_Display_Product_Metabox();
        
        // Load the API Key manager
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-api-key.php';
        
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-settings.php';

        require_once AR_DISPLAY_PLUGIN_DIR . 'includes/class-ardisplay-api-key-helper.php';
        AR_Display_api_key_Helper::init();
        $this->api_key = new AR_Display_Api_Key();
        $this->settings = new AR_Display_Settings();
        
        // Load the 3D model manager
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-3d-model.php';
        new AR_Display_3D_Model();
        
        // Add AJAX handlers
        add_action('wp_ajax_ardisplay_test_api_connection', array($this, 'test_api_connection'));
        add_action('enqueue_scripts', array($this, 'enqueue_scripts'));
        add_filter('script_loader_tag', array($this, 'add_module_type_attribute'), 10, 2);


    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url(__FILE__) . '../assets/css/ardisplay-admin.css',
            array(),
            $this->version,
            'all'
        );
        
        if (isset($_GET['page']) && $_GET['page'] === 'ardisplay-configure-ar') {
            // assets\css\admin-configure-ar.css
            wp_enqueue_style(
                $this->plugin_name . '-admin-configure-ar',
                plugin_dir_url(__FILE__) . '../assets/css/admin-configure-ar.css',
                array(),
                $this->version
            );
        }
    }

    public function add_module_type_attribute($tag, $handle) {
        // Add type="module" to specific script handles
        $module_scripts = array('three-setup', 'create-usdz-glb', 'image-to-models');
        
        if (in_array($handle, $module_scripts)) {
            $tag = str_replace(' src', ' type="module" src', $tag);
        }
        return $tag;
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        // Enqueue main plugin script
        wp_enqueue_script(
            $this->plugin_name,
            AR_DISPLAY_PLUGIN_URL . '/assets/js/ardisplay-admin.js',
            array('jquery'),
            $this->version,
            false
        );

        $screen = get_current_screen();

        if (isset($_GET['page']) && $_GET['page'] === 'ardisplay-configure-ar') {
            
            // Add import map
            $import_map = '<script type="importmap">
            {
                "imports": {
                    "three": "https://unpkg.com/three@0.174.0/build/three.module.js",
                    "three/examples/jsm/loaders/GLTFLoader.js": "https://unpkg.com/three@0.174.0/examples/jsm/loaders/GLTFLoader.js",
                    "three/examples/jsm/loaders/DRACOLoader.js": "https://unpkg.com/three@0.174.0/examples/jsm/loaders/DRACOLoader.js"
                }
            }
            </script>';
            wp_add_inline_script('es-module-shims', $import_map);

            // Enqueue es-module-shims polyfill
            wp_enqueue_script(
                'es-module-shims',
                'https://unpkg.com/es-module-shims@1.5.0/dist/es-module-shims.js',
                array(),
                '1.5.0',
                false
            );

            // Enqueue a meshoptimizer decoder
            wp_enqueue_script(
                'meshopt-decoder',
                'https://unpkg.com/meshoptimizer@0.22.0/meshopt_decoder.js',
                array(),
                '0.22.0',
                true
            );
            
            // Enqueue three-setup module
            wp_enqueue_script(
                'three-setup',
                AR_DISPLAY_PLUGIN_URL . '/assets/js/admin-configure-ar/three-setup.mjs',
                array(),
                '0.174.0',
                true
            );

            // Enqueue dependent scripts using three-setup
            wp_enqueue_script(
                'create-usdz-glb',
                AR_DISPLAY_PLUGIN_URL . '/assets/js/admin-configure-ar/create-usdz-model.js',
                array('three-setup'),
                '1.0.0',
                true
            );

            wp_enqueue_style('dropzone', 'https://unpkg.com/dropzone@5/dist/min/dropzone.min.css');
            wp_enqueue_script('dropzone', 'https://unpkg.com/dropzone@5/dist/min/dropzone.min.js', array(), '5.0.0', true);
            wp_enqueue_script('simple-crop', AR_DISPLAY_PLUGIN_URL . 'assets/js/admin-configure-ar/simple-crop.js', array(), '1.0.0', true);
            wp_enqueue_script(
                'image-to-models',
                AR_DISPLAY_PLUGIN_URL . 'assets/js/admin-configure-ar/image-to-models.js',
                array('three-setup'),
                '1.0.0',
                true
            );
        }

        // Enqueue table-specific scripts when on the product list page
        if (isset($_GET['page']) && $_GET['page'] === 'display-products-page') {
            wp_enqueue_script(
                $this->plugin_name . '-table-js',
                plugin_dir_url(__FILE__) . 'js/ardisplay-table.js',
                array('jquery'),
                $this->version,
                true
            );

            wp_localize_script($this->plugin_name . '-table-js', 'ardisplayAdminData', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'secretKey'  => class_exists('AR_Display_api_key_Helper') ? AR_Display_api_key_Helper::get_api_key() : '', // Add check
                'apiBaseUrl' => ARDISPLAY_API_ENDPOINT,
                'nonce'   => wp_create_nonce('ardisplay-toggle-ar'),
                'update_product_model_nonce' => wp_create_nonce('ardisplay-update-product-model'),
                'clear_product_model_nonce' => wp_create_nonce('ardisplay-clear-product-model')
            ));
            
        }

        // For settings page
        if (isset($screen->id) && strpos($screen->id, 'ardisplay') !== false) {
            // Add color picker
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');
            
            // Add QR code styling library
            wp_enqueue_script('qr-code-styling', 'https://unpkg.com/qr-code-styling@1.5.0/lib/qr-code-styling.js', array(), '1.5.0', true);
            
            // Add custom QR code preview script
            wp_enqueue_script('ardisplay-qr-preview', plugin_dir_url(__FILE__) . '../assets/js/ardisplay-qr-preview.js', array('jquery', 'qr-code-styling'), $this->version, true);
            
            // Add media uploader scripts
            wp_enqueue_media();
        }
    }
    
    /**
     * Add menu items to the admin area.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {
        // Add top level menu item
        add_menu_page(
            'AR Display',          // Page title
            'AR Display',                      // Menu title
            'manage_options',                 // Capability
            'ardisplay',                     // Menu slug
            array($this, 'display_home_page'), // Callback function
            'dashicons-visibility',           // Icon
            76                               // Position
        );
        
        // Add submenu items (these will appear under ardisplay)
        add_submenu_page(
            'ardisplay',                     // Parent slug
            'Products',                      // Page title
            'Products',                      // Menu title
            'manage_options',                 // Capability
            'display-products-page',                     // Menu slug (same as parent to make it the default)
            array($this, 'display_products_page') // Callback function
        );

        // Add 3D Models page
        add_submenu_page(
            'ardisplay',                     // Parent slug
            '3D Models',                     // Page title
            '3D Models',                     // Menu title
            'manage_options',                // Capability
            'ardisplay-models',              // Menu slug
            array($this, 'display_models_page') // Callback function
        );

        // Add Configure AR page
        add_submenu_page(
            'ardisplay',                     // Parent slug
            'Configure ARt Preview',          // Page title
            'Configure ARt Preview',          // Menu title
            'manage_options',                 // Capability
            'ardisplay-configure-ar',        // Menu slug
            array($this, 'display_configure_ar_page') // Callback function
        );
        
        // Add AR Button Settings page
        add_submenu_page(
            'ardisplay',                     // Parent slug
            'Settings',            // Page title
            'Settings',               // Menu title
            'manage_options',                // Capability
            'ardisplay-button-settings',     // Menu slug
            array($this, 'display_settings_page') // Callback function
        );

        // Add License page
        add_submenu_page(
            'ardisplay',                     // Parent slug
            'API Integration',                       // Page title
            'API Integration',                       // Menu title
            'manage_options',                // Capability
            'ardisplay-license',             // Menu slug
            array($this->api_key, 'display_api_key_page') // Callback function
        );
    }
    
    /**
     * Render the home page.
     *
     * @since    1.0.0
     */
    public function display_home_page() {
        include_once AR_DISPLAY_PLUGIN_DIR . 'admin/partials/ardisplay-admin-home.php';
        if (function_exists('ardisplay_home_admin_page')) {
            ardisplay_home_admin_page();
        }
    }

    /**
     * Render the products list page.
     *
     * @since    1.0.0
     */
    public function display_products_page() {
        include_once AR_DISPLAY_PLUGIN_DIR . 'admin/partials/ardisplay-admin-products.php';
    }
    
    /**
     * Render the configure AR page.
     *
     * @since    1.0.0
     */
    public function display_configure_ar_page() {
        include_once AR_DISPLAY_PLUGIN_DIR . 'admin/partials/ardisplay-admin-configure-ar/index.php';
    }
    
    /**
     * Render the settings page.
     *
     * @since    1.0.0
     */
    public function display_settings_page() {
        include_once AR_DISPLAY_PLUGIN_DIR . 'admin/partials/ardisplay-admin-settings/index.php';
    }

    /**
     * Render the 3D models page.
     *
     * @since    1.0.0
     */
    public function display_models_page() {
        include_once AR_DISPLAY_PLUGIN_DIR . 'admin/partials/ardisplay-admin-models.php';
    }
    
    /**
     * Test the API connection via AJAX
     *
     * @since    1.0.0
     */
    public function test_api_connection() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ardisplay_test_api')) {
            wp_send_json_error(__('Security check failed', 'ardisplay'));
        }
        
        // Get API settings
        $api_endpoint = get_option('ardisplay_backend_api_endpoint', '');
        $api_key = get_option('ardisplay_backend_api_key', '');
        
        if (empty($api_endpoint)) {
            wp_send_json_error(__('API endpoint not configured.', 'ardisplay'));
        }
        
        // Send a test request to the API
        $response = wp_remote_post($api_endpoint . '/test', array(
            'method' => 'GET',
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $api_key
            )
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code >= 200 && $response_code < 300) {
            wp_send_json_success(__('Connection successful!', 'ardisplay'));
        } else {
            $error_msg = sprintf(__('Error: Server returned code %d', 'ardisplay'), $response_code);
            wp_send_json_error($error_msg);
        }
    }



    public function init_ajax() {
        add_action('wp_ajax_ardisplay_test_api', array($this, 'test_api_connection'));
       }


}
