<?php
/**
 * License functionality for the ardisplay plugin
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    ardisplay
 * @subpackage ardisplay/admin
 */

/**
 * The admin-specific license functionality of the plugin.
 *
 * Handles the admin-facing aspects of license management, including
 * saving and clearing the license key through AJAX requests.
 *
 * @package    ardisplay
 * @subpackage ardisplay/admin
 * @author     ardisplay Team <info@ardisplay.com>
 */
class AR_Display_Api_Key {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * License key option name.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $api_key_option    The option name for the license key.
	 */
	private $api_key_option;
	
	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name       Optional. The name of this plugin.
	 * @param    string    $version           Optional. The version of this plugin.
	 */
	public function __construct( $plugin_name = 'ardisplay', $version = '1.0.0' ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		
		$this->api_key_option = 'ardisplay_api_key';
		
		// Register AJAX actions
		add_action( 'wp_ajax_ardisplay_save_api_key', array( $this, 'save_api_key' ) );
	}
	
	/**
	 * Display the license admin page.
	 *
	 * @since    1.0.0
	 */
	public function display_api_key_page() {
		include_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/ardisplay-admin-api-key.php';
	}
	
	/**
	 * Handle the AJAX request to save the license key.
	 *
	 * @since    1.0.0
	 */
	public function save_api_key() {
		// Check nonce for security
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'ardisplay_api_key_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'ardisplay' ) ) );
		}
		
		// // Check if license key is provided
		if ( ! isset( $_POST['api_key'] ) || empty( $_POST['api_key'] ) ) {
			wp_send_json_error( array( 'message' => __( 'Please enter a valid license key.', 'ardisplay' ) ) );
		}
		
		$api_key = sanitize_text_field( $_POST['api_key'] );

		$response = AR_Display_api_key_Helper::make_api_request('secret-key',['token'=>$api_key] , 'GET');
		
		if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
            return;
        }

		// Check if the API key is valid
		if (!isset($response['valid']) || !$response['valid']) {
			wp_send_json_error(array('message' => __( 'Invalid license key.', 'ardisplay' )));
			return;
		}

		// Save the license key
		update_option( $this->api_key_option, $api_key );
		
		wp_send_json_success( array( 
			'message' => __( 'License key saved successfully!', 'ardisplay' )
		) );
	}
	
	
	
	/**
	 * Get the license key from the database.
	 *
	 * @since    1.0.0
	 * @return   string    The license key or empty string if not set.
	 */
	public function get_api_key() {
		return AR_Display_api_key_Helper::get_api_key();
	}
	
	/**
	 * Check if a license key is set.
	 *
	 * @since    1.0.0
	 * @return   boolean    True if a license key is set, false otherwise.
	 */
	public function has_api_key() {
		return AR_Display_api_key_Helper::has_api_key();
	}
}
