<?php
/**
 * AR Display Product Metabox
 * 
 * Adds metabox to WooCommerce product editor for AR display settings
 *
 * @since      1.0.0
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

class AR_Display_Product_Metabox {

    /**
     * Initialize the class
     */
    public function __construct() {
        // Add meta box to the product editing screen
        add_action('add_meta_boxes', array($this, 'add_ar_metabox'));
        
        // Save product meta data
        add_action('woocommerce_process_product_meta', array($this, 'save_ar_product_meta'));
    }

    /**
     * Add AR Display metabox to product editor
     */
    public function add_ar_metabox() {
        add_meta_box(
            'ardisplay_product_meta',
            __('ARt Preview Settings', 'ardisplay'),
            array($this, 'render_ar_metabox'),
            'product',
            'normal',
            'default'
        );
    }

    /**
     * Render AR Display metabox content
     */
    public function render_ar_metabox($post) {
        // Add nonce for security
        wp_nonce_field('ardisplay_save_product_meta', 'ardisplay_product_meta_nonce');
        
        // Get existing values
        $ar_enabled = get_post_meta($post->ID, '_ardisplay_enabled', true);
        $model_url = get_post_meta($post->ID, '_ardisplay_model_url', true);
        $model_type = get_post_meta($post->ID, '_ardisplay_model_type', true);
        $model_id = get_post_meta($post->ID, '_ardisplay_model_id', true);
        $ar_scale = get_post_meta($post->ID, '_ardisplay_scale', true) ?: '1.0';
        ?>
        <div class="ardisplay-meta-section">
            <p>
                <label>
                    <input type="checkbox" name="ardisplay_enabled" value="yes" <?php checked($ar_enabled, 'yes'); ?> />
                    <strong><?php _e('Enable AR Preview for this product', 'ardisplay'); ?></strong>
                </label>
            </p>
            <p class="description">
                <?php _e('Check this box to enable Augmented Reality preview for this product.', 'ardisplay'); ?>
            </p>

            <?php if (!empty($model_url)): ?>
            <div class="ardisplay-model-info">
                <h4><?php _e('Current 3D Model', 'ardisplay'); ?></h4>
                <p>
                    <strong><?php _e('URL:', 'ardisplay'); ?></strong> 
                    <a href="<?php echo esc_url($model_url); ?>" target="_blank"><?php echo esc_html(basename($model_url)); ?></a>
                </p>
                <p>
                    <strong><?php _e('Type:', 'ardisplay'); ?></strong> 
                    <?php echo esc_html(strtoupper($model_type)); ?>
                </p>
                <p>
                    <strong><?php _e('Scale:', 'ardisplay'); ?></strong> 
                    <?php echo esc_html($ar_scale); ?>
                </p>
                <?php if (!empty($model_id)): ?>
                <p>
                    <strong><?php _e('Backend Model ID:', 'ardisplay'); ?></strong> 
                    <input type="text" id="ardisplay_model_id" name="ardisplay_model_id" value="<?php echo esc_attr($model_id); ?>" class="regular-text" readonly />
                </p>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <div class="ardisplay-configure-link" style="margin-top: 15px;">
                <?php 
                // Show configuration link if the product has already been saved
                if ($post->ID > 0) {
                    $configure_url = admin_url('admin.php?page=ardisplay-configure-ar&product_id=' . $post->ID);
                    ?>
                    <a href="<?php echo esc_url($configure_url); ?>" class="button">
                        <?php _e('Configure ARt Preview', 'ardisplay'); ?>
                    </a>
                    <p class="description">
                        <?php _e('Upload or update the 3D model for this product.', 'ardisplay'); ?>
                    </p>
                    <?php
                } else {
                    ?>
                    <p class="description">
                        <?php _e('Save the product first to configure AR preview settings.', 'ardisplay'); ?>
                    </p>
                    <?php
                }
                ?>
            </div>
        </div>
        <?php
    }

    /**
     * Save AR product meta
     */
    public function save_ar_product_meta($post_id) {
        // Check if nonce is set
        if (!isset($_POST['ardisplay_product_meta_nonce'])) {
            return;
        }
        
        // Verify nonce
        if (!wp_verify_nonce($_POST['ardisplay_product_meta_nonce'], 'ardisplay_save_product_meta')) {
            return;
        }
        
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save data
        $ar_enabled = isset($_POST['ardisplay_enabled']) ? 'yes' : 'no';
        update_post_meta($post_id, '_ardisplay_enabled', $ar_enabled);
        
        if (isset($_POST['ardisplay_model_url'])) {
            update_post_meta($post_id, '_ardisplay_model_url', sanitize_url($_POST['ardisplay_model_url']));
        }
        
        if (isset($_POST['ardisplay_model_type'])) {
            update_post_meta($post_id, '_ardisplay_model_type', sanitize_text_field($_POST['ardisplay_model_type']));
        }
        
        if (isset($_POST['ardisplay_scale'])) {
            update_post_meta($post_id, '_ardisplay_scale', (float) $_POST['ardisplay_scale']);
        }
        
        if (isset($_POST['ardisplay_model_id'])) {
            update_post_meta($post_id, '_ardisplay_model_id', sanitize_text_field($_POST['ardisplay_model_id']));
        }
    }
}
