/**
 * ardisplay Table JavaScript
 * Handles toggle switches and table interactions
 */
(function($) {
    'use strict';

    // Ensure ardisplayAdminData is defined
    if (typeof ardisplayAdminData === 'undefined') {
        console.error('ardisplayAdminData is not defined. Please check the script localization.');
        return;
    }

    // Initialize when document is ready
    $(document).ready(function() {

        // Toggle switch for AR enablement
        $('.ar-toggle').on('change', function() {
            const $this = $(this);
            const productId = $this.data('product-id');
            const nonce = $this.data('nonce');
            const isEnabled = $this.prop('checked');
            
            // Show loading indicator
            $this.closest('label').addClass('loading');
            
            // Make AJAX call to update AR status
            $.ajax({
                url: ardisplayAdminData.ajax_url,
                type: 'POST',
                data: {
                    action: 'ardisplay_toggle_ar',
                    product_id: productId,
                    enabled: isEnabled,
                    nonce: ardisplayAdminData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Display success message
                        ardisplayToastrSuccess(response.data.message);
                    } else {
                        // Revert toggle if there was an error
                        $this.prop('checked', !$this.prop('checked'));
                        ardisplayToastrError(response.data.message || 'An error occurred');
                    }
                },
                error: function() {
                    // Revert toggle on error
                    $this.prop('checked', !$this.prop('checked'));
                    ardisplayToastrError('Failed to update AR status');
                },
                complete: function() {
                    // Remove loading state
                    $this.closest('label').removeClass('loading');
                }
            });
        });
        
        // Apply modern styling to WP tables
        enhanceTableAppearance();
    });
    
    // Function to enhance table appearance with animations and effects
    function enhanceTableAppearance() {
        // Add staggered entrance animation to rows
        $('.wp-list-table tbody tr').each(function(index) {
            $(this).css({
                'animation': 'fadeInUp 0.3s ease forwards',
                'animation-delay': (index * 0.05) + 's',
                'opacity': '0'
            });
        });
                
        // Make image cells visually engaging
        $('.wp-list-table .column-image img').addClass('product-thumbnail');
    }
    
    // Add CSS for notifications
    $('<style>')
        .text(`
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(10px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            .product-thumbnail {
                border-radius: 5px;
                transition: all 0.3s ease;
                box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            }
            
            .product-thumbnail:hover {
                transform: scale(1.1);
                box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            }
            .ardisplay-switch.loading .ardisplay-slider {
                background-color: #ccc !important;
                animation: pulse 1.5s infinite;
            }
            
            @keyframes pulse {
                0% {
                    opacity: 1;
                }
                50% {
                    opacity: 0.5;
                }
                100% {
                    opacity: 1;
                }
            }
            
            .hover-effect {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                border-radius: 50%;
                background: rgba(255,255,255,0.1);
                transform: scale(0);
                opacity: 0;
                transition: all 0.3s ease;
            }
            
            .ardisplay-configure-button:hover .hover-effect {
                transform: scale(1.5);
                opacity: 1;
            }
        `)
        .appendTo('head');

    // Function to open model selection modal
    function openModelSelectionModal(productId) {
        // Create modal HTML
        const modalHtml = `
            <div id="ardisplay-model-modal" class="ardisplay-modal">
                <div class="ardisplay-modal-content">
                    <span class="ardisplay-close-modal">&times;</span>
                    <h2>Select 3D Model</h2>
                    <div id="ardisplay-model-list" class="ardisplay-model-grid">
                        <!-- Models will be dynamically populated here -->
                        <div class="ardisplay-loading">Loading models...</div>
                    </div>
                    <div class="ardisplay-modal-actions">
                        <button id="ardisplay-cancel-model" class="button">Cancel</button>
                    </div>
                </div>
            </div>
        `;

        // Append modal to body
        jQuery('body').append(modalHtml);

        // // Fetch available models
        // jQuery.ajax({
        //     url: ardisplayAdminData.ajax_url,
        //     method: 'POST',
        //     data: {
        //         action: 'ardisplay_fetch_models',
        //         nonce: ardisplayAdminData.fetch_models_nonce
        //     },
        //     success: function(response) {
        //         if (response.success) {
        //             const modelList = jQuery('#ardisplay-model-list');
        //             modelList.empty();

        //             response.data.models.forEach(model => {
        //                 const modelItem = `
        //                     <div class="ardisplay-model-item" data-model-id="${model.id}">
        //                         <img src="${model.poster}" alt="${model.title}">
        //                         <h3>${model.title}</h3>
        //                         <button class="ardisplay-select-model" data-model-id="${model.id}">Select</button>
        //                     </div>
        //                 `;
        //                 modelList.append(modelItem);
        //             });

        //             // Attach select model event
        //             jQuery('.ardisplay-select-model').on('click', function() {
        //                 const modelId = jQuery(this).data('model-id');
        //                 updateProductModel(productId, modelId);
        //             });
        //         } else {
        //             console.error('Failed to fetch models:', response.data.message);
        //         }
        //     },
        //     error: function() {
        //         console.error('Error fetching models');
        //     }
        // });

        // Close modal events
        jQuery('.ardisplay-close-modal, #ardisplay-cancel-model').on('click', function() {
            jQuery('#ardisplay-model-modal').remove();
        });
    }

    // Function to update product model
    function updateProductModel(productId, modelId) {
        jQuery.ajax({
            url: ardisplayAdminData.ajax_url,
            method: 'POST',
            data: {
                action: 'ardisplay_update_product_model',
                product_id: productId,
                model_id: modelId,
                nonce: ardisplayAdminData.update_product_model_nonce
            },
            success: function(response) {
                if (response.success) {
                    // Close modal and refresh page or update UI
                    jQuery('#ardisplay-model-modal').remove();
                    location.reload(); // Simple refresh, can be replaced with more dynamic update
                } else {
                    console.error('Failed to update model:', response.data.message);
                }
            },
            error: function() {
                console.error('Error updating product model');
            }
        });
    }

    // Function to clear product model
    function clearProductModel(productId) {
        jQuery.ajax({
            url: ardisplayAdminData.ajax_url,
            method: 'POST',
            data: {
                action: 'ardisplay_clear_product_model',
                product_id: productId,
                nonce: ardisplayAdminData.clear_product_model_nonce
            },
            success: function(response) {
                if (response.success) {
                    location.reload(); // Simple refresh, can be replaced with more dynamic update
                } else {
                    console.error('Failed to clear model:', response.data.message);
                }
            },
            error: function() {
                console.error('Error clearing product model');
            }
        });
    }

    // Add event listeners for change and clear model buttons
    jQuery(document).ready(function($) {
        $('.ardisplay-change-model').on('click', function(e) {
            e.preventDefault();
            const productId = $(this).data('product-id');
            openModelSelectionModal(productId);
        });

        $('.ardisplay-clear-model').on('click', function(e) {
            e.preventDefault();
            const productId = $(this).data('product-id');
            clearProductModel(productId);
        });
    });

})(jQuery);
