<?php
/**
 * Settings admin page
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    Ardisplay
 * @subpackage Ardisplay/admin/partials
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

require_once AR_DISPLAY_PLUGIN_DIR . 'assets/js/upload-handlers.js.php';

// Include settings class if not already included
if (!class_exists('AR_Display_Settings') && defined('AR_DISPLAY_PLUGIN_DIR')) {
    require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-settings.php';
}

// Enqueue required styles and scripts
if (function_exists('wp_enqueue_style')) {
    wp_enqueue_style('wp-color-picker');
    
    // Enqueue the admin settings CSS file
    $css_url = AR_DISPLAY_PLUGIN_URL . 'admin/partials/ardisplay-admin-settings/css/admin-settings.css';
    wp_enqueue_style('ardisplay-admin-settings', $css_url, array(), '1.0.1');
}
// 
if (function_exists('wp_enqueue_script')) {
    wp_enqueue_script('wp-color-picker');
}


// Get the current active tab
$active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'button';

$model_handler = new AR_Display_Settings();
$api_response = $active_tab !== 'product' ? $model_handler->get_settings() : null;

// Check if $api_response is a WP_Error
if (is_wp_error($api_response)) {
    // Log the error
    error_log('AR Display Settings Error: ' . $api_response->get_error_message());
    
    // Set default settings
    $api_response = array(
        'button' => array(
            'btnText' => 'View in 3D',
            'btnBgColor' => '#000000',
            'btnTextColor' => '#ffffff',
            'cornerRadius' => 4,
            'btnSize' => 16,
            'btnIcon' => 'none'
        ),
        'qr' => array(
            'website' => (function_exists('get_site_url') ? get_site_url() : 'https://example.com'),
            'redirectUrl' => '',
            'QRsize' => 150,
            'position' => 'top-right',
            'margin' => 10,
            'dotStyle' => 'square',
            'dotColor' => '#000000',
            'cornerStyle' => 'square',
            'cornerColor' => '#000000',
            'cornerDotStyle' => 'dot',
            'cornerDotColor' => '#000000',
            'backgroundColor' => '#ffffff',
            'imgBackground' => true,
            'faviconUrl' => '',
            'faviconId' => '',
            'faviconMargin' => '3'
        ),
        'display' => array(
            'glbee' => 'popup',
            'qrCode' => 'popup',
            'autoRotate' => true,
            'zoomAllowed' => true,
            'showARButton' => true
        )
        
    );
}

// Parse API response for settings
$button_settings = array(
    'text' => isset($api_response['button']['btnText']) ? $api_response['button']['btnText'] : 'View in 3D',
    'backgroundColor' => isset($api_response['button']['btnBgColor']) ? $api_response['button']['btnBgColor'] : '#000000',
    'textColor' => isset($api_response['button']['btnTextColor']) ? $api_response['button']['btnTextColor'] : '#ffffff',
    'borderRadius' => isset($api_response['button']['cornerRadius']) ? intval($api_response['button']['cornerRadius']) : 4,
    'fontSize' => isset($api_response['button']['btnSize']) ? intval($api_response['button']['btnSize']) : 16,
    'icon' => isset($api_response['button']['btnIcon']) ? $api_response['button']['btnIcon'] : 'none',
    'position' => 'below' // Default position since it's not in the API response
);

// Ensure all required keys exist with defaults
$button_settings = (function_exists('wp_parse_args') ? wp_parse_args($button_settings, array(
    'text' => 'View in 3D',
    'backgroundColor' => '#000000',
    'textColor' => '#ffffff',
    'borderRadius' => 4,
    'fontSize' => 16,
    'icon' => 'none',
    'position' => 'below'
)) : array_merge(array(
    'text' => 'View in 3D',
    'backgroundColor' => '#000000',
    'textColor' => '#ffffff',
    'borderRadius' => 4,
    'fontSize' => 16,
    'icon' => 'none',
    'position' => 'below'
), $button_settings));

$qr_settings = array(
    'baseUrl' => isset($api_response['qr']['website']) ? $api_response['qr']['website'] : (function_exists('get_site_url') ? get_site_url() : 'https://example.com'),
    'redirectUrl' => isset($api_response['qr']['redirectUrl']) ? $api_response['qr']['redirectUrl'] : '',
    'size' => isset($api_response['qr']['QRsize']) ? intval($api_response['qr']['QRsize']) : 150,
    'position' => isset($api_response['qr']['position']) ? $api_response['qr']['position'] : 'top-right',
    'margin' => isset($api_response['qr']['margin']) ? intval($api_response['qr']['margin']) : 10,
    'dotStyle' => isset($api_response['qr']['dotStyle']) ? $api_response['qr']['dotStyle'] : 'square',
    'dotColor' => isset($api_response['qr']['dotColor']) ? $api_response['qr']['dotColor'] : '#000000',
    'cornerStyle' => isset($api_response['qr']['cornerStyle']) ? $api_response['qr']['cornerStyle'] : 'square',
    'cornerColor' => isset($api_response['qr']['cornerColor']) ? $api_response['qr']['cornerColor'] : '#000000',
    'cornerDotStyle' => isset($api_response['qr']['cornerDotStyle']) ? $api_response['qr']['cornerDotStyle'] : 'dot',
    'cornerDotColor' => isset($api_response['qr']['cornerDotColor']) ? $api_response['qr']['cornerDotColor'] : '#000000',
    'backgroundColor' => isset($api_response['qr']['backgroundColor']) ? $api_response['qr']['backgroundColor'] : '#ffffff',
    'imgBackground' => isset($api_response['qr']['imgBackground']) ? $api_response['qr']['imgBackground'] : true,
    'faviconUrl' => isset($api_response['qr']['faviconUrl']) ? $api_response['qr']['faviconUrl'] : '',
    'faviconId' => isset($api_response['qr']['faviconId']) ? $api_response['qr']['faviconId'] : '',
    'faviconMargin' => isset($api_response['qr']['faviconMargin']) ? $api_response['qr']['faviconMargin'] : '3'
);

$qr_settings = (function_exists('wp_parse_args') ? wp_parse_args($qr_settings, array(
    'baseUrl' => (function_exists('get_site_url') ? get_site_url() : 'https://example.com'),
    'redirectUrl' => '',
    'size' => 150,
    'position' => 'top-right',
    'margin' => 10,
    'dotStyle' => 'square',
    'dotColor' => '#000000',
    'cornerStyle' => 'square',
    'cornerColor' => '#000000',
    'cornerDotStyle' => 'dot',
    'cornerDotColor' => '#000000',
    'backgroundColor' => '#ffffff',
    'imgBackground' => true,
    'faviconUrl' => '',
    'faviconId' => '',
    'faviconMargin' => '3'
)) : array_merge(array(
    'baseUrl' => (function_exists('get_site_url') ? get_site_url() : 'https://example.com'),
    'redirectUrl' => '',
    'size' => 150,
    'position' => 'top-right',
    'margin' => 10,
    'dotStyle' => 'square',
    'dotColor' => '#000000',
    'cornerStyle' => 'square',
    'cornerColor' => '#000000',
    'cornerDotStyle' => 'dot',
    'cornerDotColor' => '#000000',
    'backgroundColor' => '#ffffff',
    'imgBackground' => true,
    'faviconUrl' => '',
    'faviconId' => '',
    'faviconMargin' => '3'
), $qr_settings));

$display_settings = array(
    'mode' => isset($api_response['mode']['glbee']) ? $api_response['mode']['glbee'] : 'popup',
    'qrCode' => isset($api_response['mode']['qrCode']) ? $api_response['mode']['qrCode'] : 'popup',
    'autoRotate' => true,
    'zoomAllowed' => true,
    'showARButton' => true
);

$display_settings = (function_exists('wp_parse_args') ? wp_parse_args($display_settings, array(
    'mode' => 'popup',
    'qrCode' => 'popup',
    'autoRotate' => true,
    'zoomAllowed' => true,
    'showARButton' => true
)) : array_merge(array(
    'mode' => 'popup',
    'qrCode' => 'popup',
    'autoRotate' => true,
    'zoomAllowed' => true,
    'showARButton' => true
), $display_settings));

$product_button_settings = array(
    'position' => get_option('ardisplay_display_position', 'before_add_to_cart'),
    'fullWidth' => (bool)get_option('ardisplay_display_full_width', 0)
);


if (function_exists('wp_localize_script')) {
    wp_localize_script('ardisplay-qr-preview', 'ardisplayQrSettings', $qr_settings);
}


?>

<div class="wrap ardisplay-settings-wrapper">
    <h1><?php echo esc_html__('Ardisplay Settings', 'ardisplay'); ?></h1>
    
    
    <?php if (function_exists('settings_errors')) { settings_errors('ardisplay_settings'); } ?>
    
    <h2 class="nav-tab-wrapper">
        <a href="?page=ardisplay-button-settings&tab=button" class="nav-tab <?php echo $active_tab === 'button' ? 'nav-tab-active' : ''; ?>">
            <?php esc_html_e('Button', 'ardisplay'); ?>
        </a>
        <a href="?page=ardisplay-button-settings&tab=qr" class="nav-tab <?php echo $active_tab === 'qr' ? 'nav-tab-active' : ''; ?>">
            <?php esc_html_e('QR Code', 'ardisplay'); ?>
        </a>
        <a href="?page=ardisplay-button-settings&tab=display" class="nav-tab <?php echo $active_tab === 'display' ? 'nav-tab-active' : ''; ?>">
            <?php esc_html_e('Display Mode', 'ardisplay'); ?>
        </a>

        <a href="?page=ardisplay-button-settings&tab=product" class="nav-tab <?php echo $active_tab === 'product' ? 'nav-tab-active' : ''; ?>">
            <?php esc_html_e('Product Button', 'ardisplay'); ?>
        </a>
    </h2>
    
    <div class="ardisplay-admin-box">
        <?php if ($active_tab === 'button'): ?>
            <!-- Button Customization Tab -->
            <div id="button-settings" class="ardisplay-tab-content">
                <div class="ardisplay-settings-container">
                    <h2><?php esc_html_e('Button Settings', 'ardisplay'); ?></h2>
                    
                    <div id="ardisplay-button-settings-wrapper" style="display: flex; flex-wrap: wrap; gap: 30px; align-items: flex-start;">
                        <!-- Form Section - Left Column -->
                        <div class="ardisplay-form-container" style="flex: 1; min-width: 50%; max-width: 65%;">
                            <form method="post" action="" id="button-settings-form" name="ardisplay_button_settings_form">
                                <?php if (function_exists('wp_nonce_field')) { wp_nonce_field('ardisplay_button_settings_nonce', 'ardisplay_button_settings_nonce'); } ?>
                                <input type="hidden" name="ardisplay_button_settings_submit" value="1">
                                
                                <table class="form-table">
                                    <tr>
                                        <th scope="row">
                                            <label for="button_text"><?php esc_html_e('Label', 'ardisplay'); ?></label>
                                        </th>
                                        <td>
                                            <input type="text" id="button_text" name="button_text" class="regular-text" value="<?php echo esc_attr($button_settings['text']); ?>">
                                            <p class="description"><?php esc_html_e('This is the text on the button in your website.', 'ardisplay'); ?></p>
                                        </td>
                                    </tr>
                                    
                                    <tr>
                                        <th scope="row">
                                            <label for="button_font_size"><?php esc_html_e('Size', 'ardisplay'); ?></label>
                                        </th>
                                        <td>
                                            <select id="button_font_size" name="button_font_size">
                                                <option value="18" <?php selected($button_settings['fontSize'], 18); ?>><?php esc_html_e('None', 'ardisplay'); ?></option>
                                                <option value="21" <?php selected($button_settings['fontSize'], 21); ?>><?php esc_html_e('Small', 'ardisplay'); ?></option>
                                                <option value="24" <?php selected($button_settings['fontSize'], 24); ?>><?php esc_html_e('Medium', 'ardisplay'); ?></option>
                                                <option value="27" <?php selected($button_settings['fontSize'], 27); ?>><?php esc_html_e('Large', 'ardisplay'); ?></option>
                                            </select>
                                            <p class="description"><?php esc_html_e('This is the text size on the button in your website.', 'ardisplay'); ?></p>
                                        </td>
                                    </tr>
                                    
                                    <tr>
                                        <th scope="row">
                                            <label for="button_text_color"><?php esc_html_e('Text Color', 'ardisplay'); ?></label>
                                        </th>
                                        <td>
                                            <div class="color-picker-container">
                                                <input type="color" id="button_text_color" name="button_text_color" value="<?php echo esc_attr($button_settings['textColor']); ?>">
                                                <span class="color-value"><?php echo esc_attr($button_settings['textColor']); ?></span>
                                            </div>
                                            <p class="description"><?php esc_html_e('This is the text color on the button in your website.', 'ardisplay'); ?></p>
                                        </td>
                                    </tr>
                                    
                                    <tr>
                                        <th scope="row">
                                            <label for="button_bg_color"><?php esc_html_e('Background Color', 'ardisplay'); ?></label>
                                        </th>
                                        <td>
                                            <div class="color-picker-container">
                                                <input type="color" id="button_bg_color" name="button_bg_color" value="<?php echo esc_attr($button_settings['backgroundColor']); ?>">
                                                <span class="color-value"><?php echo esc_attr($button_settings['backgroundColor']); ?></span>
                                            </div>
                                            <p class="description"><?php esc_html_e('This is the background color on the button in your website.', 'ardisplay'); ?></p>
                                        </td>
                                    </tr>
                                    
                                    <tr>
                                        <th scope="row">
                                            <label for="button_icon"><?php esc_html_e('Icon', 'ardisplay'); ?></label>
                                        </th>
                                        <td>
                                            <select id="button_icon" name="button_icon">
                                                <option value="Eye" <?php selected(isset($button_settings['icon']) ? $button_settings['icon'] : 'Eye', 'Eye'); ?>><?php esc_html_e('Eye', 'ardisplay'); ?></option>
                                                <option value="Rotate3D" <?php selected(isset($button_settings['icon']) ? $button_settings['icon'] : '', 'Rotate3D'); ?>><?php esc_html_e('Rotate 3D', 'ardisplay'); ?></option>
                                                <option value="Box" <?php selected(isset($button_settings['icon']) ? $button_settings['icon'] : '', 'Box'); ?>><?php esc_html_e('Box', 'ardisplay'); ?></option>
                                                <option value="FileAxis3D" <?php selected(isset($button_settings['icon']) ? $button_settings['icon'] : '', 'FileAxis3D'); ?>><?php esc_html_e('File Axis 3D', 'ardisplay'); ?></option>
                                                <option value="Scan" <?php selected(isset($button_settings['icon']) ? $button_settings['icon'] : '', 'Scan'); ?>><?php esc_html_e('Scan', 'ardisplay'); ?></option>
                                                <option value="Blocks" <?php selected(isset($button_settings['icon']) ? $button_settings['icon'] : '', 'Blocks'); ?>><?php esc_html_e('Blocks', 'ardisplay'); ?></option>
                                            </select>
                                            <p class="description"><?php esc_html_e('This is the icon on the button in your website.', 'ardisplay'); ?></p>
                                        </td>
                                    </tr>
                                    
                                    <tr>
                                        <th scope="row">
                                            <label for="button_border_radius"><?php esc_html_e('Corner Radius', 'ardisplay'); ?></label>
                                        </th>
                                        <td>
                                            <select id="button_border_radius" name="button_border_radius">
                                                <option value="0" <?php selected($button_settings['borderRadius'], 0); ?>><?php esc_html_e('None', 'ardisplay'); ?></option>
                                                <option value="4" <?php selected($button_settings['borderRadius'], 4); ?>><?php esc_html_e('Small', 'ardisplay'); ?></option>
                                                <option value="8" <?php selected($button_settings['borderRadius'], 8); ?>><?php esc_html_e('Medium', 'ardisplay'); ?></option>
                                                <option value="9999" <?php selected($button_settings['borderRadius'], 9999); ?>><?php esc_html_e('Full', 'ardisplay'); ?></option>
                                            </select>
                                            <p class="description"><?php esc_html_e('This is the corner radius of the button in your website.', 'ardisplay'); ?></p>
                                        </td>
                                    </tr>
                                </table>
                                
                                <?php if (function_exists('submit_button')) { submit_button(__('Save Button Settings', 'ardisplay'), 'primary', 'submit', true, array('id' => 'submit')); } ?>
                            </form>
                        </div>
                        
                        <!-- Preview Section - Right Column -->
                        <div class="ardisplay-qr-preview-container" style="flex: 0 0 300px; position: sticky; top: 32px; align-self: flex-start; background: #fff; padding: 25px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05); margin-bottom: 20px;">
                        <h3 style="margin-top: 0; border-bottom: 2px solid #2271b1; padding-bottom: 12px; color: #2271b1;"><?php esc_html_e('Button Preview', 'ardisplay'); ?></h3>
                        <div class="ardisplay-qr-preview-wrapper" style="flex-direction: column;">
                        <div class="button-preview-container position-<?php echo isset($button_settings['position']) ? esc_attr($button_settings['position']) : 'below'; ?>">
                                    <button id="button-preview" style="padding: 10px 20px; font-size: <?php echo isset($button_settings['fontSize']) ? esc_attr($button_settings['fontSize']) : '16'; ?>px; color: <?php echo isset($button_settings['textColor']) ? esc_attr($button_settings['textColor']) : '#ffffff'; ?>; background-color: <?php echo isset($button_settings['backgroundColor']) ? esc_attr($button_settings['backgroundColor']) : '#000000'; ?>; border: none; border-radius: <?php echo isset($button_settings['borderRadius']) ? esc_attr($button_settings['borderRadius']) : '4'; ?>px; cursor: pointer; display: inline-flex; align-items: center; gap: 8px;">
                                        <span id="button-icon-preview" style="display: <?php echo isset($button_settings['icon']) && $button_settings['icon'] !== 'none' ? 'inline-block' : 'none'; ?>;">
                                            <?php 
                                            $icon = isset($button_settings['icon']) ? $button_settings['icon'] : 'none';
                                            switch ($icon) {
                                                case 'Eye':
                                                    echo '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-eye"><path d="M2.062 12.348a1 1 0 0 1 0-.696 10.75 10.75 0 0 1 19.876 0 1 1 0 0 1 0 .696 10.75 10.75 0 0 1-19.876 0"></path><circle cx="12" cy="12" r="3"></circle></svg>';
                                                    break;
                                                case 'Rotate3D':
                                                    echo '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-rotate3d"><path d="M16.466 7.5C15.643 4.237 13.952 2 12 2 9.239 2 7 6.477 7 12s2.239 10 5 10c.342 0 .677-.069 1-.2"></path><path d="m15.194 13.707 3.814 1.86-1.86 3.814"></path><path d="M19 15.57c-1.804.885-4.274 1.43-7 1.43-5.523 0-10-2.239-10-5s4.477-5 10-5c4.838 0 8.873 1.718 9.8 4"></path></svg>';
                                                    break;
                                                case 'Box':
                                                    echo '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-box"><path d="M21 8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16Z"></path><path d="m3.3 7 8.7 5 8.7-5"></path><path d="M12 22V12"></path></svg>';
                                                    break;
                                                case 'FileAxis3D':
                                                    echo '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-file-axis3d"><path d="M7.52 21.48C4.25 19.94 1.91 16.76 1.55 13H0.05C0.56 19.16 5.71 24 12 24L12.66 23.97L8.85 20.16L7.52 21.48M19.95 11C19.44 4.84 14.29 0 8 0L7.34 0.03L11.15 3.84L12.48 2.52C15.75 4.07 18.09 7.25 18.45 11H19.95Z"/><path d="M12 6C8.69 6 6 8.69 6 12C6 15.31 8.69 18 12 18C15.31 18 18 15.31 18 12C18 8.69 15.31 6 12 6Z"/></svg>';
                                                    break;
                                                case 'Scan':
                                                    echo '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-scan"><path d="M3 7V5a2 2 0 0 1 2-2h2"></path><path d="M17 3h2a2 2 0 0 1 2 2v2"></path><path d="M21 17v2a2 2 0 0 1-2 2h-2"></path><path d="M7 21H5a2 2 0 0 1-2-2v-2"></path></svg>';
                                                    break;
                                                case 'Blocks':
                                                    echo '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-blocks"><rect width="7" height="7" x="14" y="3" rx="1"></rect><path d="M10 21V8a1 1 0 0 0-1-1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-5a1 1 0 0 0-1-1H3"></path></svg>';
                                                    break;
                                                default:
                                                    echo '';
                                                    break;
                                            }
                                            ?>
                                        </span>
                                        <span id="button-text-preview"><?php echo esc_html($button_settings['text']); ?></span>
                                    </button>
                                </div>
                        </div>
                </div>
            </div>
        <?php elseif ($active_tab === 'qr'): ?>
            <!-- QR Code Customization Tab -->
            <div id="ardisplay-qr-tab" class="ardisplay-tab-content">
                <h2><?php esc_html_e('QR Code Customization', 'ardisplay'); ?></h2>
                
                <div id="ardisplay-settings-wrapper" style="display: flex; flex-wrap: wrap; gap: 30px; align-items: flex-start;">
                    <!-- Form Section - Left Column -->
                    <div class="ardisplay-form-container" style="flex: 1; min-width: 50%; max-width: 65%;">
                    <form method="post" action="" id="ardisplay_qr_settings_form" name="ardisplay_qr_settings_form">
                        <?php if (function_exists('wp_nonce_field')) { wp_nonce_field('ardisplay_qr_settings_nonce', 'ardisplay_qr_settings_nonce'); } ?>
                        <input type="hidden" name="ardisplay_qr_settings_submit" value="1">
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_base_url"><?php esc_html_e('Website URL', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="url" id="ardisplay_qr_base_url" name="ardisplay_qr_base_url" value="<?php echo esc_url($qr_settings['baseUrl']); ?>" class="regular-text">
                                    <p class="description"><?php esc_html_e('The base URL for your website. Used to generate QR codes.', 'ardisplay'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_redirect_url"><?php esc_html_e('Redirect URL (optional)', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="url" id="ardisplay_qr_redirect_url" name="ardisplay_qr_redirect_url" value="<?php echo esc_url($qr_settings['redirectUrl']); ?>" class="regular-text">
                                    <p class="description"><?php esc_html_e('Optional URL to redirect users scanning QR codes. Leave empty to use the product page URL.', 'ardisplay'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_size"><?php esc_html_e('QR Code Size', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="number" id="ardisplay_qr_size" name="ardisplay_qr_size" value="<?php echo esc_attr($qr_settings['size']); ?>" min="50" max="1000" step="10" class="small-text">
                                    <p class="description"><?php esc_html_e('Size of the QR code in pixels (width and height).', 'ardisplay'); ?></p>
                                </td>
                            </tr>
                            <th scope="row">
                                <label for="ardisplay_qr_position"><?php esc_html_e('QR Code Display', 'ardisplay'); ?></label>
                            </th>
                            <td>
                                <select id="ardisplay_qr_position" name="ardisplay_qr_position">
                                    <option value="top-right" <?php selected($qr_settings['position'], 'top-right'); ?>><?php esc_html_e('Top right', 'ardisplay'); ?></option>
                                    <option value="top-left" <?php selected($qr_settings['position'], 'top-left'); ?>><?php esc_html_e('Top left', 'ardisplay'); ?></option>
                                    <option value="bottom-right" <?php selected($qr_settings['position'], 'bottom-right'); ?>><?php esc_html_e('Bottom right', 'ardisplay'); ?></option>
                                    <option value="bottom-left" <?php selected($qr_settings['position'], 'bottom-left'); ?>><?php esc_html_e('Bottom left', 'ardisplay'); ?></option>
                                </select>
                            </td>
                            
                            <tr>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_margin"><?php esc_html_e('QR Code Margin', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="number" id="ardisplay_qr_margin" name="ardisplay_qr_margin" value="<?php echo esc_attr($qr_settings['margin']); ?>" min="0" max="100" step="1" class="small-text">
                                    <p class="description"><?php esc_html_e('Margin around the QR code in pixels.', 'ardisplay'); ?></p>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_dot_style"><?php esc_html_e('Dot Style', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <select id="ardisplay_qr_dot_style" name="ardisplay_qr_dot_style">
                                        <option value="square" <?php selected($qr_settings['dotStyle'], 'square'); ?>><?php esc_html_e('Square', 'ardisplay'); ?></option>
                                        <option value="dots" <?php selected($qr_settings['dotStyle'], 'dots'); ?>><?php esc_html_e('Dot', 'ardisplay'); ?></option>
                                        <option value="rounded" <?php selected($qr_settings['dotStyle'], 'rounded'); ?>><?php esc_html_e('Rounded', 'ardisplay'); ?></option>
                                        <option value="classy" <?php selected($qr_settings['dotStyle'], 'classy'); ?>><?php esc_html_e('Classy', 'ardisplay'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_dot_color"><?php esc_html_e('Dot Color', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="color" id="ardisplay_qr_dot_color" name="ardisplay_qr_dot_color" value="<?php echo esc_attr($qr_settings['dotColor']); ?>">
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_corner_style"><?php esc_html_e('Corner Style', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <select id="ardisplay_qr_corner_style" name="ardisplay_qr_corner_style">
                                        <option value="square" <?php selected($qr_settings['cornerStyle'], 'square'); ?>><?php esc_html_e('Square', 'ardisplay'); ?></option>
                                        <option value="dot" <?php selected($qr_settings['cornerStyle'], 'dot'); ?>><?php esc_html_e('Dot', 'ardisplay'); ?></option>
                                        <option value="circle" <?php selected($qr_settings['cornerStyle'], 'circle'); ?>><?php esc_html_e('Circle', 'ardisplay'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_corner_color"><?php esc_html_e('Corner Color', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="color" id="ardisplay_qr_corner_color" name="ardisplay_qr_corner_color" value="<?php echo esc_attr($qr_settings['cornerColor']); ?>">
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_corner_dot_style"><?php esc_html_e('Corner Dot Style', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <select id="ardisplay_qr_corner_dot_style" name="ardisplay_qr_corner_dot_style">
                                        <option value="square" <?php selected($qr_settings['cornerDotStyle'], 'square'); ?>><?php esc_html_e('Square', 'ardisplay'); ?></option>
                                        <option value="dot" <?php selected($qr_settings['cornerDotStyle'], 'dot'); ?>><?php esc_html_e('Dot', 'ardisplay'); ?></option>
                                        <option value="circle" <?php selected($qr_settings['cornerDotStyle'], 'circle'); ?>><?php esc_html_e('Circle', 'ardisplay'); ?></option>
                                    </select>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_corner_dot_color"><?php esc_html_e('Corner Dot Color', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="color" id="ardisplay_qr_corner_dot_color" name="ardisplay_qr_corner_dot_color" value="<?php echo esc_attr($qr_settings['cornerDotColor']); ?>">
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_background_color"><?php esc_html_e('Background Color', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="color" id="ardisplay_qr_background_color" name="ardisplay_qr_background_color" value="<?php echo esc_attr($qr_settings['backgroundColor']); ?>">
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_img_background"><?php esc_html_e('Image Background', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <label>
                                        <input type="checkbox" id="ardisplay_qr_img_background" name="ardisplay_qr_img_background" <?php checked($qr_settings['imgBackground'], true); ?>>
                                        <?php esc_html_e('Use image background', 'ardisplay'); ?>
                                    </label>
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_favicon_id"><?php esc_html_e('Favicon Image', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <div class="ardisplay-file-upload">
                                        <input type="file" id="ardisplay_qr_favicon_upload" name="ardisplay_qr_favicon_upload" />
                                        <label for="ardisplay_qr_favicon_upload" class="ardisplay-file-upload-button">Choose File</label>
                                        <span class="ardisplay-file-name">No file selected</span>
                                    </div>
                                </td>
                            </tr>
                            <tr hidden>
                                <th scope="row">
                                    <label for="ardisplay_qr_favicon_url"><?php esc_html_e('Favicon URL', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="url" id="ardisplay_qr_favicon_url" name="ardisplay_qr_favicon_url" value="<?php echo esc_url($qr_settings['faviconUrl']); ?>" class="regular-text">
                                </td>
                            </tr>
                            
                            <tr hidden>
                                <th scope="row">
                                    <label for="ardisplay_qr_favicon_id"><?php esc_html_e('Favicon ID', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="text" id="ardisplay_qr_favicon_id" name="ardisplay_qr_favicon_id" value="<?php echo esc_attr($qr_settings['faviconId']); ?>" class="regular-text">
                                </td>
                            </tr>
                            
                            <tr>
                                <th scope="row">
                                    <label for="ardisplay_qr_favicon_margin"><?php esc_html_e('Favicon Margin', 'ardisplay'); ?></label>
                                </th>
                                <td>
                                    <input type="number" id="ardisplay_qr_favicon_margin" name="ardisplay_qr_favicon_margin" value="<?php echo esc_attr($qr_settings['faviconMargin']); ?>" min="0" max="100" step="1" class="small-text">
                                </td>
                            </tr>
                        </table>
                        
                        <?php if (function_exists('submit_button')) { submit_button(__('Save QR Code Settings', 'ardisplay')); } ?>
                    </form>
                    </div>
                    
                    <!-- QR Code Preview Section - Right Column -->
                    <div class="ardisplay-qr-preview-container" style="flex: 0 0 300px; position: sticky; top: 32px; align-self: flex-start; background: #fff; padding: 25px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05); margin-bottom: 20px;">
                        <h3 style="margin-top: 0; border-bottom: 2px solid #2271b1; padding-bottom: 12px; color: #2271b1;"><?php esc_html_e('QR Code Preview', 'ardisplay'); ?></h3>
                        <div class="ardisplay-qr-preview-wrapper" style="flex-direction: column;">
                            <div id="ardisplay-qr-preview" style="margin: 15px auto; min-height: 250px; min-width: 250px; border: 1px dashed #ddd; padding: 15px; border-radius: 5px; background-color: #f9f9f9; transition: all 0.3s ease;"></div>
                        </div>
                    </div>
                </div>
            </div>
        <?php elseif ($active_tab === 'display'): ?>
            <!-- Display Mode Tab -->
            <div id="display-mode-settings" class="ardisplay-tab-content">
                <div class="ardisplay-settings-container">
                    <h2><?php esc_html_e('Display Mode Settings', 'ardisplay'); ?></h2>
                    
                    <div class="ardisplay-form-container">
                        <form method="post" action="" id="ardisplay_display_settings_form" name="ardisplay_display_settings_form">
                            <?php if (function_exists('wp_nonce_field')) { wp_nonce_field('ardisplay_display_settings_nonce', 'ardisplay_display_settings_nonce'); } ?>
                            <input type="hidden" name="ardisplay_display_settings_submit" value="1">
                            
                            <table class="form-table">
                                <tr>
                                    <th scope="row">
                                        <label for="ardisplay_display_mode"><?php esc_html_e('Display Mode', 'ardisplay'); ?></label>
                                    </th>
                                    <td>
                                        <select id="ardisplay_display_mode" name="ardisplay_display_mode">
                                            <option value="popup" <?php selected($display_settings['mode'], 'popup'); ?>><?php esc_html_e('Popup', 'ardisplay'); ?></option>
                                            <option value="inpage" <?php selected($display_settings['mode'], 'inpage'); ?>><?php esc_html_e('In Page', 'ardisplay'); ?></option>
                                            <option value="none" <?php selected($display_settings['mode'], 'none'); ?>><?php esc_html_e('None', 'ardisplay'); ?></option>
                                        </select>
                                        <p class="description"><?php esc_html_e('Choose how 3D models will be displayed on your site.', 'ardisplay'); ?></p>
                                    </td>
                                </tr>
                            </table>
                            
                            <?php if (function_exists('submit_button')) { submit_button(__('Save Display Mode Settings', 'ardisplay')); } ?>
                        </form>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <!-- Display Product Button Tab -->
            <div id="display-product-settings" class="ardisplay-tab-content">
                <div class="ardisplay-settings-container">
                    <h2><?php esc_html_e('Display Product Button Settings', 'ardisplay'); ?></h2>
                    
                    <div class="ardisplay-form-container">
                        <form method="post" action="" id="ardisplay_product_button_settings_form" name="ardisplay_product_button_settings_form">
                            <?php if (function_exists('wp_nonce_field')) { wp_nonce_field('ardisplay_product_button_settings_nonce', 'ardisplay_product_button_settings_nonce'); } ?>
                            <input type="hidden" name="ardisplay_product_button_settings_submit" value="1">
                            <table class="form-table">
                                <tr>
                                    <th scope="row">
                                        <label for="ardisplay_display_position"><?php esc_html_e('Position', 'ardisplay'); ?></label>
                                    </th>
                                    <td>
                                        <select id="ardisplay_display_position" name="ardisplay_display_position">
                                            <option value="after_add_to_cart" <?php selected($product_button_settings['position'], 'after_add_to_cart'); ?>><?php esc_html_e('After Add to Cart', 'ardisplay'); ?></option>
                                            <option value="before_add_to_cart" <?php selected($product_button_settings['position'], 'before_add_to_cart'); ?>><?php esc_html_e('Before Add to Cart', 'ardisplay'); ?></option>
                                            <option value="before_image" <?php selected($product_button_settings['position'], 'before_image'); ?>><?php esc_html_e('Before Image', 'ardisplay'); ?></option>
                                        </select>
                                        <p class="description"><?php esc_html_e('Choose where the 3D model button will be displayed on the product page.', 'ardisplay'); ?></p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row">
                                        <label for="ardisplay_display_full_width"><?php esc_html_e('Full Width', 'ardisplay'); ?></label>
                                    </th>
                                    <td>
                                        <label for="ardisplay_display_full_width">
                                            <input type="checkbox" id="ardisplay_display_full_width" name="ardisplay_display_full_width" value="1" <?php checked($product_button_settings['fullWidth'], 1); ?>>
                                            <?php esc_html_e('Make the button full width.', 'ardisplay'); ?>
                                        </label>
                                    </td>
                                </tr>
                            </table>
                            
                            <?php if (function_exists('submit_button')) { submit_button(__('Save Display Mode Settings', 'ardisplay')); } ?>
                        </form>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    $('.ardisplay-color-picker').wpColorPicker();
});
</script>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Function to update the button preview
    function updateButtonPreview() {
        var buttonText = $('#button_text').val() || 'View in AR';
        var buttonBgColor = $('#button_bg_color').val();
        var buttonTextColor = $('#button_text_color').val();
        var buttonBorderRadius = $('#button_border_radius').val() + 'px';
        var buttonFontSize = $('#button_font_size').val();
        var buttonIcon = $('#button_icon').val();
        
        // Calculate padding based on font size
        var paddingVertical = (parseInt(buttonFontSize) / 2) + 'px';
        var paddingHorizontal = buttonFontSize + 'px';
        
        // Update the button text
        $('#button-text-preview').text(buttonText);
        
        // Update the button styles
        $('#button-preview').css({
            'background-color': buttonBgColor,
            'color': buttonTextColor,
            'border-radius': buttonBorderRadius,
            'font-size': (parseInt(buttonFontSize) - 6) + 'px',
            'padding': paddingVertical + ' ' + paddingHorizontal
        });
        
        // Update icon display
        if (buttonIcon && buttonIcon !== 'none') {
            $('#button-icon-preview').show();
            
            // Update SVG icon based on selection
            var iconSvg = '';
            switch (buttonIcon) {
                case 'Eye':
                    iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-eye"><path d="M2.062 12.348a1 1 0 0 1 0-.696 10.75 10.75 0 0 1 19.876 0 1 1 0 0 1 0 .696 10.75 10.75 0 0 1-19.876 0"></path><circle cx="12" cy="12" r="3"></circle></svg>';
                    break;
                case 'Rotate3D':
                    iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-rotate3d"><path d="M16.466 7.5C15.643 4.237 13.952 2 12 2 9.239 2 7 6.477 7 12s2.239 10 5 10c.342 0 .677-.069 1-.2"></path><path d="m15.194 13.707 3.814 1.86-1.86 3.814"></path><path d="M19 15.57c-1.804.885-4.274 1.43-7 1.43-5.523 0-10-2.239-10-5s4.477-5 10-5c4.838 0 8.873 1.718 9.8 4"></path></svg>';
                    break;
                case 'Box':
                    iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-box"><path d="M21 8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16Z"></path><path d="m3.3 7 8.7 5 8.7-5"></path><path d="M12 22V12"></path></svg>';
                    break;
                case 'FileAxis3D':
                    iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-file-axis3d"><path d="M7.52 21.48C4.25 19.94 1.91 16.76 1.55 13H0.05C0.56 19.16 5.71 24 12 24L12.66 23.97L8.85 20.16L7.52 21.48M19.95 11C19.44 4.84 14.29 0 8 0L7.34 0.03L11.15 3.84L12.48 2.52C15.75 4.07 18.09 7.25 18.45 11H19.95Z"/><path d="M12 6C8.69 6 6 8.69 6 12C6 15.31 8.69 18 12 18C15.31 18 18 15.31 18 12C18 8.69 15.31 6 12 6Z"/></svg>';
                    break;
                case 'Scan':
                    iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-scan"><path d="M3 7V5a2 2 0 0 1 2-2h2"></path><path d="M17 3h2a2 2 0 0 1 2 2v2"></path><path d="M21 17v2a2 2 0 0 1-2 2h-2"></path><path d="M7 21H5a2 2 0 0 1-2-2v-2"></path></svg>';
                    break;
                case 'Blocks':
                    iconSvg = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="21" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-blocks"><rect width="7" height="7" x="14" y="3" rx="1"></rect><path d="M10 21V8a1 1 0 0 0-1-1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1v-5a1 1 0 0 0-1-1H3"></path></svg>';
                    break;
                default:
                    iconSvg = '';
                    break;
            }
            $('#button-icon-preview').html(iconSvg);
        } else {
            $('#button-icon-preview').hide();
        }
    }
    
    // Call the update function on page load
        updateButtonPreview();
    
    // Call the update function when any input changes
    $('#button_text, #button_bg_color, #button_text_color, #button_border_radius, #button_font_size, #button_icon').on('input change', function() {
        updateButtonPreview();
    });
    
    // Update color value displays
    $('#button_text_color, #button_bg_color').on('input change', function() {
        var colorValue = $(this).val();
        $(this).next('.color-value').text(colorValue);
    });
    
    // Generic function to update settings via AJAX
    function updateSettingsViaAjax(settingsType, settingsData) {
        var saveButton = $('#submit');
        var originalText = saveButton.val();
        
        // Disable the button and show loading state
        saveButton.prop('disabled', true).val('Saving...');
        
        // Prepare the data
        var data = {
            'action': 'ardisplay_update_settings',
            'nonce': '<?php echo wp_create_nonce("ardisplay_settings_nonce"); ?>',
            'settings': settingsData
        };
        
        // Send the AJAX request
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: data,
            success: function(response) {
                // Re-enable the button
                saveButton.prop('disabled', false).val(originalText);
                
                if (response.success) {
                    // Show success message with toastr
                    ardisplayToastrSuccess('Settings saved successfully!');
                } else {
                    // Show error message with toastr
                    ardisplayToastrError(response.data.message || 'Failed to save settings. Please try again.');
                }
            },
            error: function(xhr, status, error) {
                // Re-enable the button
                saveButton.prop('disabled', false).val(originalText);
                
                // Log the error
                console.error('AJAX Error:', status, error);
                
                // Show error message with toastr
                ardisplayToastrError('Failed to connect to the server. Please check your connection and try again.');
            }
        });
    }
    
    // Example of how to use the AJAX function with the button settings form
    $('#button-settings-form').on('submit', function(e) {
        e.preventDefault(); // Prevent the default form submission
        
        // Collect form data
        var buttonSettings = {
            btnText: $('#button_text').val() || 'View in AR',
            btnBgColor: $('#button_bg_color').val() || '#000000',
            btnTextColor: $('#button_text_color').val() || '#ffffff',
            cornerRadius: $('#button_border_radius').val() || '4',
            btnSize: $('#button_font_size').val() || '16',
            btnIcon: $('#button_icon').val() || 'none'
        };

        // Call the AJAX function with form=button
        var data = {
            form: 'button',
            ...buttonSettings
        };

        updateSettingsViaAjax('button', data);
    });
    
    // QR Code Settings form AJAX submission
    $('form[name="ardisplay_qr_settings_form"]').on('submit', function(e) {
        e.preventDefault();
        
        // Collect form data using the API structure
        var qrSettings = {
            form: 'qr',
            faviconId: $('#ardisplay_qr_favicon_id').val() || '',
            faviconUrl: $('#ardisplay_qr_favicon_url').val() || '',
            faviconMargin: $('#ardisplay_qr_favicon_margin').val() || '3',
            QRsize: $('#ardisplay_qr_size').val() || '150',
            dotStyle: $('#ardisplay_qr_dot_style').val() || 'square',
            dotColor: $('#ardisplay_qr_dot_color').val() || '#000000',
            cornerStyle: $('#ardisplay_qr_corner_style').val() || 'square',
            cornerColor: $('#ardisplay_qr_corner_color').val() || '#000000',
            cornerDotStyle: $('#ardisplay_qr_corner_dot_style').val() || 'square',
            cornerDotColor: $('#ardisplay_qr_corner_dot_color').val() || '#000000',
            backgroundColor: $('#ardisplay_qr_background_color').val() || '#ffffff',
            imgBackground: $('#ardisplay_qr_img_background').is(':checked'),
            website: $('#ardisplay_qr_base_url').val() || window.location.origin
        };
        
        // Call the AJAX function
        updateSettingsViaAjax('qr', qrSettings);
    });
    
    // Display Mode Settings form AJAX submission
    $('form[name="ardisplay_display_settings_form"]').on('submit', function(e) {
        e.preventDefault();
        
        // Collect form data using the API structure
        var displaySettings = {
            form: 'mode',
            glbee: $('#ardisplay_display_mode').val() || 'none',
        };
        
        // Call the AJAX function
        updateSettingsViaAjax('display', displaySettings);
    });

    // Product Button Settings form AJAX submission
    $('form[name="ardisplay_product_button_settings_form"]').on('submit', function(e) {
        e.preventDefault();
        
        // Collect form data using the API structure
        var productButtonSettings = {
            form: 'product',
            position: $('#ardisplay_display_position').val() || 'before_add_to_cart',
            fullWidth: $('#ardisplay_display_full_width').is(':checked'),
        };
        
        // Call the AJAX function
        updateSettingsViaAjax('product', productButtonSettings);
    });
});
</script>
