
const THREE = window.THREE;
const GLTFLoader = window.GLTFLoader;
const DRACOLoader = window.DRACOLoader;
const USDZExporter = window.USDZExporter;
const GLTFExporter = window.GLTFExporter;

// Define frame styles globally
window.FRAME_STYLES = [
  {
    id: "modern-white",
    name: "Modern White",
    color: 0xffffff,
    metalness: 0.8,
    roughness: 0.2,
    thickness: 0.03,
  },
  {
    id: "classic-white",
    name: "Classic White",
    color: 0xffffff,
    metalness: 0,
    roughness: 0.9,
    thickness: 0.03,
  },
  {
    id: "modern-black",
    name: "Modern Black",
    color: 0x000000,
    metalness: 0.8,
    roughness: 0.2,
    thickness: 0.03,
  },
];

// Define other functions globally
window.loadTexture = function (url) {
  return new Promise((resolve, reject) => {
    const loader = new THREE.TextureLoader();
    loader.load(
      url,
      (texture) => {
        resolve(texture);
      },
      undefined,
      (err) => {
        let msg = "Error loading texture";
        if (typeof err === "object" && err !== null && "message" in err) {
          msg = String(err.message);
        }
        reject(new Error(msg));
      }
    );
  });
};

// Add utility functions at the top level
window.cmToMeters = function (cmValue) {
  // Handle both string (with "cm" suffix) and number inputs
  const numericValue =
    typeof cmValue === "string"
      ? parseFloat(cmValue.replace("cm", ""))
      : parseFloat(cmValue);
  return numericValue / 100;
};

window.metersToString = function (meters) {
  return (meters * 100).toFixed(2);
};

class FrameBuilder {
  // Constructor properties declared without types or modifiers.
  constructor(root, frameStyle, width, height) {
    this.root = root;
    this.frameStyle = frameStyle;
    this.dimensions = { width, height };
  }

  createFrameMaterial() {
    return new THREE.MeshStandardMaterial({
      color: this.frameStyle.color,
      metalness: this.frameStyle.metalness,
      roughness: this.frameStyle.roughness,
    });
  }

  addFramePart(geometry, position) {
    const mesh = new THREE.Mesh(geometry, this.createFrameMaterial());
    mesh.position.copy(position);
    this.root.add(mesh);
  }

  buildFrame() {
    const { width, height } = this.dimensions;
    const { thickness } = this.frameStyle;

    // Left Frame
    this.addFramePart(
      new THREE.BoxGeometry(thickness, height + 2 * thickness, thickness),
      new THREE.Vector3(-width / 2 - thickness / 2, 0, 0)
    );

    // Right Frame
    this.addFramePart(
      new THREE.BoxGeometry(thickness, height + 2 * thickness, thickness),
      new THREE.Vector3(width / 2 + thickness / 2, 0, 0)
    );

    // Top Frame
    this.addFramePart(
      new THREE.BoxGeometry(width + 2 * thickness, thickness, thickness),
      new THREE.Vector3(0, height / 2 + thickness / 2, 0)
    );

    // Bottom Frame
    this.addFramePart(
      new THREE.BoxGeometry(width + 2 * thickness, thickness, thickness),
      new THREE.Vector3(0, -height / 2 - thickness / 2, 0)
    );
  }
}

window.createGLBModel = async function (
  texture,
  frameStyle = window.FRAME_STYLES[0]
) {
  return new Promise((resolve, reject) => {
    const scene = new THREE.Scene();
    const image = texture.image;
    const aspectRatio = image.width / image.height;
    const planeHeight = 1;
    const planeWidth = planeHeight * aspectRatio;

    // Create image plane
    const planeGeometry = new THREE.PlaneGeometry(planeWidth, planeHeight);
    const planeMaterial = new THREE.MeshStandardMaterial({
      map: texture,
      side: THREE.DoubleSide,
      transparent: true,
    });

    const plane = new THREE.Mesh(planeGeometry, planeMaterial);
    scene.add(plane);

    // Build frame
    const frameBuilder = new FrameBuilder(
      scene,
      frameStyle,
      planeWidth,
      planeHeight
    );
    frameBuilder.buildFrame();

    // Add light
    const light = new THREE.AmbientLight(0xffffff, 1);
    scene.add(light);

    // Export to GLB with proper binary handling
    const exporter = new GLTFExporter();
    exporter.parse(
      scene,
      (gltf) => {
        if (gltf instanceof ArrayBuffer) {
          resolve(gltf);
        } else if (gltf instanceof Uint8Array) {
          resolve(gltf.buffer);
        } else {
          // Fallback: convert JSON string to binary buffer.
          const jsonString = JSON.stringify(gltf);
          const encoder = new TextEncoder();
          resolve(encoder.encode(jsonString).buffer);
        }
      },
      (error) => {
        reject(new Error(JSON.stringify(error)));
      },
      {
        binary: true,
        onlyVisible: true,
        maxTextureSize: 4096,
        animations: [],
        includeCustomExtensions: false,
      }
    );
  });
};

window.createUSDZModel = async function (
  texture,
  frameStyle = window.FRAME_STYLES[0],
  anchoringAlignment = "horizontal"
) {
  return new Promise((resolve, reject) => {
    const scene = new THREE.Scene();
    const image = texture.image;
    const aspectRatio = image.width / image.height;
    const planeHeight = 1;
    const planeWidth = planeHeight * aspectRatio;

    // Create a container group
    const container = new THREE.Group();
    scene.add(container);

    // Create image plane with front side material
    const planeGeometry = new THREE.PlaneGeometry(planeWidth, planeHeight);
    const planeMaterial = new THREE.MeshStandardMaterial({
      map: texture,
      side: THREE.FrontSide,
      transparent: true,
    });
    const plane = new THREE.Mesh(planeGeometry, planeMaterial);
    plane.position.z = 0.01;
    container.add(plane);

    // Create and add the frame group
    const frameGroup = new THREE.Group();
    const frameBuilder = new FrameBuilder(
      frameGroup,
      frameStyle,
      planeWidth,
      planeHeight
    );
    frameBuilder.buildFrame();
    container.add(frameGroup);

    if (anchoringAlignment === "vertical") {
      container.rotation.x = -Math.PI / 2;
    }

    scene.updateMatrixWorld(true);

    const usdzExportOptions = {
      ar: {
        anchoring: { type: "plane" },
        planeAnchoring: { alignment: anchoringAlignment },
      },
      includeAnchoringProperties: true,
      quickLookCompatible: false,
      maxTextureSize: 1024,
    };
    var ExporterConstructor = window.USDZExporter || THREE.USDZExporter;
    if (!ExporterConstructor) {
      return reject(new Error("USDZExporter is not available"));
    }
    var exporter = new ExporterConstructor();
    exporter.parse(
      scene,
      (usdzArrayBuffer) => {
        resolve(usdzArrayBuffer);
      },
      (error) => {
        reject(new Error(String(error)));
      },
      usdzExportOptions
    );
  });
};

// Replace getModelDimensions with a fallback implementation:
window.getModelDimensions = async function (modelViewer) {
  // Create a new GLTFLoader, and add DRACOLoader if available.
  const loader = new GLTFLoader();
  if (typeof DRACOLoader !== "undefined") {
    const dracoLoader = new DRACOLoader();
    dracoLoader.setDecoderPath(
      "https://www.gstatic.com/draco/versioned/decoders/1.5.6/"
    );
    loader.setDRACOLoader(dracoLoader);
  }
  return new Promise((resolve, reject) => {
    loader.load(
      modelViewer.src,
      (gltf) => {
        const box = new THREE.Box3().setFromObject(gltf.scene);
        const size = new THREE.Vector3();
        box.getSize(size);
        resolve({
          width: metersToString(size.x),
          height: metersToString(size.y),
          depth: metersToString(size.z),
        });
      },
      undefined,
      reject
    );
  });
};

// Update getScaleForSize function
window.getScaleForSize = function (glbUrl, size) {
  return new Promise((resolve, reject) => {
    try {
      const loader = new GLTFLoader();

      // Add MeshoptDecoder if available
      if (typeof MeshoptDecoder !== "undefined") {
        loader.setMeshoptDecoder(MeshoptDecoder);
      }

      // Add DRACOLoader if available
      if (typeof DRACOLoader !== "undefined") {
        const dracoLoader = new DRACOLoader();
        dracoLoader.setDecoderPath(
          "https://www.gstatic.com/draco/versioned/decoders/1.5.6/"
        );
        loader.setDRACOLoader(dracoLoader);
      }

      loader.load(
        glbUrl,
        (gltf) => {
          const scene = gltf.scene;
          const box = new THREE.Box3().setFromObject(scene);
          const vecSize = new THREE.Vector3();
          box.getSize(vecSize);

          // Use the global cmToMeters function
          const desiredW = window.cmToMeters(size.width);
          const desiredH = window.cmToMeters(size.height);
          const desiredD = window.cmToMeters(size.depth);

          const scaleX = desiredW / vecSize.x;
          const scaleY = desiredH / vecSize.y;
          const scaleZ = desiredD / vecSize.z;

          resolve({
            x: scaleX,
            y: scaleY,
            z: scaleZ,
          });
        },
        // Progress callback
        (xhr) => {
          console.log((xhr.loaded / xhr.total) * 100 + "% loaded");
        },
        // Error callback
        (error) => {
          console.error("Error loading GLTF:", error);
          reject(error);
        }
      );
    } catch (e) {
      console.error("Error in getScaleForSize:", e);
      reject(e);
    }
  });
};
