class SimpleCrop {
  constructor(container, options = {}) {
    // Merge options with defaults
    this.container = container;
    this.options = Object.assign(
      {
        aspect: null, // e.g. 16/9, 1 for square, etc.
        minWidth: 0,
        minHeight: 0,
        onChange: () => {},
        onComplete: () => {},
      },
      options
    );

    this.image = container.querySelector("img");
    console.log("SimpleCrop: image element =", this.image);
    if (!this.image) {
      throw new Error("No image found in container");
    }

    // Internal state variables
    this.crop = null;
    this.mode = null; // 'create', 'move', or 'resize'
    this.resizeHandle = null; // When resizing, which handle ("nw", "n", "ne", "e", etc.)
    this.startPoint = null; // For create mode: the point where the mouse went down
    this.startMove = null; // For move mode: offset from selection top-left
    this.startResize = null; // For resize mode: the selection state on mousedown

    this.init();
  }

  init() {
    // Ensure the container can host absolutely positioned children.
    console.log(
      "SimpleCrop init: container dimensions",
      this.container.clientWidth,
      this.container.clientHeight
    );
    this.container.style.position = "relative";
    this.container.style.overflow = "hidden";
    this.container.style.userSelect = "none";

    // Create the selection element that will show the crop area.
    this.selection = document.createElement("div");
    this.selection.className = "crop-selection";
    this.selection.style.display = "none";
    this.container.appendChild(this.selection);

    // Add eight resize handles (four corners and four sides)
    this.addResizeHandles();

    // Inject necessary CSS styles.
    this.addStyles();

    // Bind events.
    this.container.addEventListener("mousedown", this.onMouseDown);
    document.addEventListener("mousemove", this.onMouseMove);
    document.addEventListener("mouseup", this.onMouseUp);

    // Set default crop selection to full image.
    // If the image is already loaded, update immediately; otherwise wait for load.
    if (this.image.complete) {
      console.log("Image complete is true");
      this.setDefaultCrop();
    } else {
      console.log("Image not complete, adding load event listener");
      this.image.addEventListener("load", () => this.setDefaultCrop());
    }
  }

  addStyles() {
    const style = document.createElement("style");
    style.textContent = `
        .crop-selection {
          position: absolute;
          border: 2px solid #fff;
          box-shadow: 0 0 0 9999px rgba(0, 0, 0, 0.5);
          box-sizing: border-box;
          cursor: move;
          z-index: 10;
        }
        .resize-handle {
          position: absolute;
          width: 10px;
          height: 10px;
          background: #fff;
          border: 1px solid #333;
          z-index: 11;
          box-sizing: border-box;
        }
        /* Corner handles */
        .resize-handle.nw { top: -5px; left: -5px; cursor: nw-resize; }
        .resize-handle.ne { top: -5px; right: -5px; cursor: ne-resize; }
        .resize-handle.sw { bottom: -5px; left: -5px; cursor: sw-resize; }
        .resize-handle.se { bottom: -5px; right: -5px; cursor: se-resize; }
        /* Side handles */
        .resize-handle.n { top: -5px; left: 50%; transform: translateX(-50%); cursor: n-resize; }
        .resize-handle.s { bottom: -5px; left: 50%; transform: translateX(-50%); cursor: s-resize; }
        .resize-handle.e { top: 50%; right: -5px; transform: translateY(-50%); cursor: e-resize; }
        .resize-handle.w { top: 50%; left: -5px; transform: translateY(-50%); cursor: w-resize; }
      `;
    document.head.appendChild(style);
  }

  addResizeHandles() {
    // Eight handles: four corners and four sides.
    const handles = ["nw", "n", "ne", "e", "se", "s", "sw", "w"];
    handles.forEach((handle) => {
      const div = document.createElement("div");
      div.className = `resize-handle ${handle}`;
      div.dataset.handle = handle;
      this.selection.appendChild(div);
    });
  }

  // ---------------- Event Handlers ----------------

  // onMouseDown – Determine what the user is doing (new creation, move, or resize)
  onMouseDown = (e) => {
    const rect = this.container.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;
    const target = e.target;

    if (target.classList.contains("resize-handle")) {
      // Start resizing an existing selection.
      this.mode = "resize";
      this.resizeHandle = target.dataset.handle;
      this.startResize = {
        x, // starting mouse coordinate
        y,
        left: this.selection.offsetLeft,
        top: this.selection.offsetTop,
        width: this.selection.offsetWidth,
        height: this.selection.offsetHeight,
      };
      e.stopPropagation();
    } else if (target === this.selection) {
      // Clicked inside an existing selection => move it.
      this.mode = "move";
      this.startMove = {
        dx: x - this.selection.offsetLeft,
        dy: y - this.selection.offsetTop,
      };
      e.stopPropagation();
    } else if (target === this.image || target === this.container) {
      // Creating a new selection.
      this.mode = "create";
      this.startPoint = { x, y };
      this.selection.style.display = "block";
      this.setSelectionStyle(x, y, 0, 0);
    }
  };

  // onMouseMove – Update the selection based on the current mode
  onMouseMove = (e) => {
    if (!this.mode) return;
    const rect = this.container.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const y = e.clientY - rect.top;

    if (this.mode === "create") {
      this.updateSelectionRectFromPoints(
        this.startPoint.x,
        this.startPoint.y,
        x,
        y
      );
    } else if (this.mode === "move") {
      const newLeft = x - this.startMove.dx;
      const newTop = y - this.startMove.dy;
      this.moveSelectionTo(newLeft, newTop);
    } else if (this.mode === "resize") {
      this.handleResizeMove(x, y);
    }

    this.updateCrop();
    e.preventDefault();
  };

  // onMouseUp – Finalize the crop and clear the current mode.
  onMouseUp = () => {
    if (this.mode) {
      this.mode = null;
      this.resizeHandle = null;
      this.startPoint = null;
      this.startMove = null;
      this.startResize = null;

      if (this.crop && typeof this.options.onComplete === "function") {
        this.options.onComplete(this.crop);
      }
    }
  };

  // ---------------- Selection Manipulation ----------------

  // Given two corner points, compute the top/left, width/height and enforce aspect ratio & boundaries.
  updateSelectionRectFromPoints(x1, y1, x2, y2) {
    let left = Math.min(x1, x2);
    let top = Math.min(y1, y2);
    let width = Math.abs(x2 - x1);
    let height = Math.abs(y2 - y1);

    // If an aspect ratio is set, adjust dimensions so that the ratio is maintained.
    if (this.options.aspect) {
      const aspect = this.options.aspect;
      if (width / height > aspect) {
        width = height * aspect;
      } else {
        height = width / aspect;
      }
      // Adjust left/top if dragging to the left/up from the start point.
      if (x2 < x1) {
        left = x1 - width;
      }
      if (y2 < y1) {
        top = y1 - height;
      }
    }

    // Keep the selection within the container.
    const containerWidth = this.container.clientWidth;
    const containerHeight = this.container.clientHeight;
    if (left + width > containerWidth) {
      width = containerWidth - left;
      if (this.options.aspect) height = width / this.options.aspect;
    }
    if (top + height > containerHeight) {
      height = containerHeight - top;
      if (this.options.aspect) width = height * this.options.aspect;
    }

    this.setSelectionStyle(left, top, width, height);
  }

  // Apply new position while keeping the selection fully inside the container.
  moveSelectionTo(newLeft, newTop) {
    const containerWidth = this.container.clientWidth;
    const containerHeight = this.container.clientHeight;
    const width = this.selection.offsetWidth;
    const height = this.selection.offsetHeight;

    newLeft = Math.max(0, Math.min(newLeft, containerWidth - width));
    newTop = Math.max(0, Math.min(newTop, containerHeight - height));

    this.setSelectionStyle(newLeft, newTop, width, height);
  }

  // Resize based on the mouse movement and starting state.
  handleResizeMove(x, y) {
    const containerWidth = this.container.clientWidth;
    const containerHeight = this.container.clientHeight;
    const start = this.startResize;
    let dx = x - start.x;
    let dy = y - start.y;
    let newLeft = start.left;
    let newTop = start.top;
    let newWidth = start.width;
    let newHeight = start.height;

    switch (this.resizeHandle) {
      case "nw":
        newLeft = start.left + dx;
        newTop = start.top + dy;
        newWidth = start.width - dx;
        newHeight = start.height - dy;
        break;
      case "ne":
        newTop = start.top + dy;
        newWidth = start.width + dx;
        newHeight = start.height - dy;
        break;
      case "sw":
        newLeft = start.left + dx;
        newWidth = start.width - dx;
        newHeight = start.height + dy;
        break;
      case "se":
        newWidth = start.width + dx;
        newHeight = start.height + dy;
        break;
      // Centered resizers: update only one dimension.
      case "n": // Center top: update height only.
        newTop = start.top + dy;
        newHeight = start.height - dy;
        newLeft = start.left;
        newWidth = start.width;
        break;
      case "s": // Center bottom: update height only.
        newHeight = start.height + dy;
        newLeft = start.left;
        newWidth = start.width;
        break;
      case "e": // Center right: update width only.
        newWidth = start.width + dx;
        newTop = start.top;
        newHeight = start.height;
        break;
      case "w": // Center left: update width only.
        newLeft = start.left + dx;
        newWidth = start.width - dx;
        newTop = start.top;
        newHeight = start.height;
        break;
      default:
        return;
    }

    // Constrain selection to container bounds.
    if (newLeft < 0) {
      newWidth += newLeft;
      newLeft = 0;
    }
    if (newTop < 0) {
      newHeight += newTop;
      newTop = 0;
    }
    if (newLeft + newWidth > containerWidth) {
      newWidth = containerWidth - newLeft;
    }
    if (newTop + newHeight > containerHeight) {
      newHeight = containerHeight - newTop;
    }

    this.setSelectionStyle(newLeft, newTop, newWidth, newHeight);
  }

  // Update the style of the selection box.
  setSelectionStyle(left, top, width, height) {
    this.selection.style.left = `${left}px`;
    this.selection.style.top = `${top}px`;
    this.selection.style.width = `${width}px`;
    this.selection.style.height = `${height}px`;
  }

  /**
   * Update the crop object in a way that takes into account that the image
   * might be displayed using object-fit: contain. In other words, we calculate
   * the actual coordinates in the image's natural dimensions.
   */
  updateCrop() {
    const containerWidth = this.container.clientWidth;
    const containerHeight = this.container.clientHeight;
    const img = this.image;
    const naturalWidth = img.naturalWidth;
    const naturalHeight = img.naturalHeight;

    // Calculate the displayed size of the image inside the container.
    const scale = Math.min(
      containerWidth / naturalWidth,
      containerHeight / naturalHeight
    );
    const displayedWidth = naturalWidth * scale;
    const displayedHeight = naturalHeight * scale;
    const offsetX = (containerWidth - displayedWidth) / 2;
    const offsetY = (containerHeight - displayedHeight) / 2;

    const selectionLeft = this.selection.offsetLeft;
    const selectionTop = this.selection.offsetTop;
    const selectionWidth = this.selection.offsetWidth;
    const selectionHeight = this.selection.offsetHeight;

    // Map the selection rectangle from container coordinates to the image's natural size.
    let x = (selectionLeft - offsetX) / scale;
    let y = (selectionTop - offsetY) / scale;
    let width = selectionWidth / scale;
    let height = selectionHeight / scale;

    // Clamp values to ensure they don't exceed the image's dimensions.
    x = Math.max(0, x);
    y = Math.max(0, y);
    if (x + width > naturalWidth) {
      width = naturalWidth - x;
    }
    if (y + height > naturalHeight) {
      height = naturalHeight - y;
    }

    this.crop = { x, y, width, height, unit: "px" };

    if (typeof this.options.onChange === "function") {
      this.options.onChange(this.crop);
    }
  }

  // Set default crop selection to cover the full displayed image - FIXED to take the whole container
  setDefaultCrop() {
    console.log("setDefaultCrop called");
    console.log(
      "Container dimensions at setDefaultCrop:",
      this.container.clientWidth,
      this.container.clientHeight
    );
    const containerWidth = this.container.clientWidth;
    const containerHeight = this.container.clientHeight;

    // Set crop selection to cover the entire container instead of just the visible image
    this.selection.style.display = "block";
    this.setSelectionStyle(10, 10, containerWidth - 20, containerHeight - 20);
    this.updateCrop();
  }

  destroy() {
    this.container.removeEventListener("mousedown", this.onMouseDown);
    document.removeEventListener("mousemove", this.onMouseMove);
    document.removeEventListener("mouseup", this.onMouseUp);
    if (this.selection && this.selection.parentNode === this.container) {
      this.container.removeChild(this.selection);
    }
  }
}

// Add a static property to store the last crop state
SimpleCrop.lastCropState = null;

// Export for CommonJS or attach to window for browsers.
if (typeof module !== "undefined" && module.exports) {
  module.exports = SimpleCrop;
} else {
  window.SimpleCrop = SimpleCrop;
}
