/**
 * ardisplay Viewer Block
 *
 * Registers a custom block that uses the ardisplay Viewer component
 */

// Import statements for WordPress blocks
const { registerBlockType } = wp.blocks;
const { InspectorControls } = wp.blockEditor;
const { PanelBody, TextControl, SelectControl, ColorPalette, ColorPicker, ToggleControl } =
  wp.components;
const { createElement, Fragment, useEffect, useRef, useState } = wp.element;

// Make sure the model-viewer web component is defined
// This ensures the custom element is registered before we use it
if (typeof customElements !== "undefined") {
  // Check if it's not already defined to avoid errors
  if (!customElements.get("ardisplay-viewer")) {
    console.log("ardisplay Viewer not yet defined, this is informational only");
  }
}

registerBlockType("ardisplay/viewer", {
  title: "AR Display Viewer",
  icon: "format-image",
  category: "common",
  attributes: {
    modelId: {
      type: "string",
      default: "67deda5afd3e092c4b4286aa",
    },
    btnText: {
      type: "string",
      default: "View in your space",
    },
    btnTextColor: {
      type: "string",
      default: null,
    },
    btnBgColor: {
      type: "string",
      default: null,
    },
    btnIcon: {
      type: "string",
      default: null,
    },
    cornerRadius: {
      type: "string",
      default: null,
    },
    btnSize: {
      type: "string",
      default: null,
    },
    fullWidth: {
      type: "boolean",
      default: true,
    },
  },

  edit: function (props) {
    const { attributes, setAttributes, className } = props;
    const {
      modelId,
      btnText,
      btnTextColor,
      btnBgColor,
      btnIcon,
      cornerRadius,
      btnSize,
      fullWidth,
    } = attributes;
    const containerRef = useRef(null);
    const [modelOptions, setModelOptions] = useState([]);
    const { apiBaseUrl, secretKey} = ardisplayModelsData;

    useEffect(() => {
      const fetchModels = async () => {
        try {
          if (secretKey) {
            const response = await fetch(`${apiBaseUrl}/3d-model/select`, {
              headers: {
                'secret-key': secretKey
              }
            });
            const data = await response.json();
            setModelOptions(data.map(model => ({
              label: model.title,
              value: model.id
            })));
          }
        } catch (error) {
          console.error('Error fetching models:', error);
        }
      };

      fetchModels();
    }, [apiBaseUrl, secretKey]);

    // Use effect to insert the model viewer after the component mounts
    useEffect(() => {

      if (containerRef.current && secretKey) {
        // Clear previous content
        containerRef.current.innerHTML = "";

        // Create the model-viewer element
        const modelViewer = document.createElement("ardisplay-viewer");

        // Set attributes
        modelViewer.setAttribute("src", modelId);
        const settings = {
          btnText,
          btnTextColor,
          btnBgColor,
          btnIcon,
          cornerRadius,
          btnSize,
        };
        
        const settingsWithoutNull = Object.entries(settings)
          .filter(([key, value]) => value !== null && value !== undefined)
          .reduce((acc, [key, value]) => {
            acc[key] = value;
            return acc;
          }, {});
        
        const settingsJson = JSON.stringify(settingsWithoutNull);
        const settingsBase64 = btoa(settingsJson);
        modelViewer.setAttribute("ar-btn-config", settingsBase64);
        modelViewer.style.width = "100%";
        modelViewer.style.height = "100% !important";
        if (fullWidth) {
          modelViewer.setAttribute("fullWidth", true);
        }

        // Append to container
        containerRef.current.appendChild(modelViewer);
      }
      else {
        // Clear previous content
        containerRef.current.innerHTML = "";
        // Create a message element
        const message = document.createElement("div");
        message.style = {
          backgroundColor: '#ffdddd',
          color: '#ff0000',
          padding: '20px',
          border: '1px solid #ff0000',
          borderRadius: '5px',
          textAlign: 'center',
          position: 'absolute',
          top: '0',
          left: '0',
          right: '0',
          zIndex: '10'
        };
        message.textContent = "Please configure the ARDisplay API key in the plugin settings.";
        containerRef.current.appendChild(message);
      }
    }, [
      modelId,
      btnText,
      btnTextColor,
      btnBgColor,
      btnIcon,
      cornerRadius,
      btnSize,
      fullWidth,
      secretKey
    ]);

    // Create settings panel
    const inspectorControls = createElement(
      InspectorControls,
      {},
      createElement(
        PanelBody,
        { title: "Model Settings", initialOpen: true },
        createElement(SelectControl, {
          label: 'Select Model',
          value: modelId,
          options: modelOptions,
          onChange: (value) => setAttributes({ modelId: value })
        }),
        createElement(TextControl, {
          label: "Model ID",
          value: modelId,
          onChange: (value) => setAttributes({ modelId: value }),
        }),
        createElement(TextControl, {
          label: "Button Text",
          value: btnText,
          onChange: (value) => setAttributes({ btnText: value }),
        }),
        //label is required for color picker

        createElement(
          "div",
          {},
          createElement("label", {}, "Text Color"),
          createElement(ColorPalette, {
            value: btnTextColor,
            onChange: (value) => setAttributes({ btnTextColor: value }),
          })
        ),
        createElement(
          "div",
          {},
          createElement("label", {}, "Background Color"),
          createElement(ColorPalette, {
            value: btnBgColor,
            onChange: (value) => setAttributes({ btnBgColor: value }),
          })
        ),
        createElement(SelectControl, {
          label: "Button Icon",
          value: btnIcon,
          options: [
            { label: "Eye", value: "Eye" },
            { label: "Rotate3D", value: "Rotate3D" },
            { label: "Box", value: "Box" },
            { label: "FileAxis3D", value: "FileAxis3D" },
            { label: "Scan", value: "Scan" },
            { label: "Blocks", value: "Blocks" },
          ],
          onChange: (value) => setAttributes({ btnIcon: value }),
        }),
        createElement(TextControl, {
          label: "Corner Radius",
          value: cornerRadius,
          onChange: (value) => setAttributes({ cornerRadius: value }),
        }),
        createElement(TextControl, {
          label: "Button Size",
          value: btnSize,
          onChange: (value) => setAttributes({ btnSize: value }),
        }),
        createElement(ToggleControl, {
          label: 'Full Width',
          checked: fullWidth,
          onChange: (value) => setAttributes({ fullWidth: value }),
        }),
      )
    );

    // Return both the inspector controls and a placeholder for our model viewer
    return createElement(
      Fragment,
      {},
      inspectorControls,
      createElement("div", {
        className: `${className} wp-block-ar-model-viewer`,
        ref: containerRef,
      })
    );
  },

  save: function (props) {
    const { attributes } = props;
    const {
      modelId,
      btnText,
      btnTextColor,
      btnBgColor,
      btnIcon,
      cornerRadius,
      btnSize,
      fullWidth,
    } = attributes;

    // Check if the secret key is available
    const secretKey = window.ardisplaySecretKey || null;

    // Always return the ardisplay-viewer element, but conditionally add attributes
    let viewerElement = createElement(
      "div",
      { className: "wp-block-ar-model-viewer" },
      createElement("ardisplay-viewer", {
        ...(modelId ? { src: modelId } : {}),
        ...(fullWidth ? { fullWidth: true } : {}),
        style: { width: "100%", height: "100% !important" },
      })
    );

    // Add the config string only if the secret key is set
    if (secretKey) {
      viewerElement.props.children.props["ar-btn-config"] = btoa(
        JSON.stringify(
          Object.fromEntries(
            Object.entries({
              btnText,
              btnTextColor,
              btnBgColor,
              btnIcon,
              cornerRadius,
              btnSize,
            }).filter(([key, value]) => value !== null)
          )
        )
      );
    }

    return viewerElement;
  },
});
