<?php
/**
 * Provides the public-facing view for the plugin
 *
 * This file is used to markup the public-facing aspects of the plugin.
 *
 * @since      1.0.0
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

// Extract shortcode attributes
$title = isset($atts['title']) ? $atts['title'] : 'AR Product Display';
$display_type = isset($atts['display_type']) ? $atts['display_type'] : 'default';
$product_ids = isset($atts['products']) ? explode(',', $atts['products']) : array();

// Determine CSS class based on display type
$display_class = 'ardisplay-container';
if ($display_type === 'special') {
    $display_class .= ' ardisplay-special';
}
?>

<div class="<?php echo esc_attr($display_class); ?>">
    <h2 class="ardisplay-title"><?php echo esc_html($title); ?></h2>
    
    <div class="ardisplay-content">
        <?php if (!empty($product_ids)): ?>
            <div class="ardisplay-products">
                <?php 
                // Loop through the specified products
                foreach ($product_ids as $product_id):
                    $product_id = intval($product_id);
                    $product = wc_get_product($product_id);
                    
                    // Skip if product doesn't exist or AR is not enabled
                    if (!$product || get_post_meta($product_id, '_ardisplay_enabled', true) !== 'yes') {
                        continue;
                    }
                    
                    // Get the model information
                    $model_url = get_post_meta($product_id, '_ardisplay_model_url', true);
                    $model_type = get_post_meta($product_id, '_ardisplay_model_type', true);
                    $model_scale = get_post_meta($product_id, '_ardisplay_scale', true) ?: '1.0';
                    $model_id = get_post_meta($product_id, '_ardisplay_model_id', true);
                    
                    // Skip if no model URL
                    if (empty($model_url)) {
                        continue;
                    }
                ?>
                    <div class="ardisplay-product" data-product-id="<?php echo esc_attr($product_id); ?>"
                         data-model-url="<?php echo esc_url($model_url); ?>"
                         data-model-type="<?php echo esc_attr($model_type); ?>"
                         data-model-scale="<?php echo esc_attr($model_scale); ?>"
                         <?php if (!empty($model_id)): ?>
                         data-model-id="<?php echo esc_attr($model_id); ?>"
                         <?php endif; ?>
                    >
                        <div class="ardisplay-product-preview">
                            <h3><?php echo esc_html($product->get_name()); ?></h3>
                            
                            <div class="ardisplay-product-image">
                                <?php echo $product->get_image('thumbnail'); ?>
                            </div>
                            
                            <div class="ardisplay-product-details">
                                <p class="ardisplay-product-price"><?php echo $product->get_price_html(); ?></p>
                                
                                <?php if ($display_type === 'detailed'): ?>
                                    <p class="ardisplay-product-description">
                                        <?php echo wp_trim_words($product->get_short_description(), 20); ?>
                                    </p>
                                <?php endif; ?>
                                
                                <button class="ardisplay-view-ar-button" 
                                        onclick="launchAR('<?php echo esc_url($model_url); ?>', '<?php echo esc_attr($model_type); ?>', <?php echo esc_attr($model_scale); ?>, '<?php echo esc_attr($model_id); ?>')">
                                    <?php _e('View in AR', 'ardisplay'); ?>
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <script>
            function launchAR(modelUrl, modelType, modelScale, modelId) {
                // This function will be implemented by your AR JavaScript library
                console.log('Launching AR with model:', modelUrl, 'Type:', modelType, 'Scale:', modelScale, 'Model ID:', modelId);
                
                if (window.ARExperienceSDK) {
                    window.ARExperienceSDK.launchViewer({
                        model: modelUrl,
                        type: modelType,
                        scale: modelScale,
                        modelId: modelId
                    });
                } else {
                    alert('AR Experience SDK not loaded. Please check your connection or try again later.');
                }
            }
            </script>
            
        <?php else: ?>
            <p><?php _e('No AR products specified. Please use the "products" attribute with product IDs.', 'ardisplay'); ?></p>
            <p><?php _e('Example: [ardisplay products="123,456"]', 'ardisplay'); ?></p>
        <?php endif; ?>
    </div>
    
    <div class="ardisplay-footer">
        <p>Powered by ARDISPLAY</p>
    </div>
</div>
